using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

namespace AirNoise.Mathlib {


    [StructLayout(LayoutKind.Sequential)]
    public class Vector3 {

        public float x;
        public float y;
        public float z;

        /**
         * Default constructor. Creates a null vector.
         */
        public Vector3() {

            x = y = z = 0.0f;
        }

        /**
         * Constructor which enables specifying the vector's three components
         */
        public Vector3(float px, float py, float pz) {

            x = px;
            y = py;
            z = pz;
        }

        /**
         * Returns the length of the vectors
         */
        public float Length() {

            return (float)Math.Sqrt(x * x + y * y + z * z);
        }

        /**
         * Normalizes the vector
         */
        public Vector3 Normalize() {

            float l = Length();

            if (l == 0) return new Vector3(0, 0, 0);

            x /= l;
            y /= l;
            z /= l;
            return new Vector3(x, y, z);
        }

        /**
         * Calculates the dot product of two vectors
         */
        public static float operator * (Vector3 v1, Vector3 v2) {

            return v1.x * v2.x + v1.y * v2.y + v1.z * v2.z;
        }

        /**
         * Calculates the cross product of two vectors
         */
        public static Vector3 operator |(Vector3 v1, Vector3 v2) {

            float tx = v1.y * v2.z - v1.z * v2.y;
            float ty = v1.z * v2.x - v1.x * v2.z;
            float tz = v1.x * v2.y - v1.y * v2.x;

            return new Vector3(tx, ty, tz);
        }

        /**
         * Adds two vectors
         */
        public static Vector3 operator +(Vector3 v1, Vector3 v2) {

            float tx = v1.x + v2.x;
            float ty = v1.y + v2.y;
            float tz = v1.z + v2.z;

            return new Vector3(tx, ty, tz);
        }

        /**
         * Subtracts a vector from another
         */
        public static Vector3 operator -(Vector3 v1, Vector3 v2) {

            float tx = v1.x - v2.x;
            float ty = v1.y - v2.y;
            float tz = v1.z - v2.z;

            return new Vector3(tx, ty, tz);

        }


        /**
         * Multiplies a vector with a given scalar
         */
        public static Vector3 operator *(Vector3 v, float value) {

            float tx = v.x * value;
            float ty = v.y * value;
            float tz = v.z * value;

            return new Vector3(tx, ty, tz);
        }


        public static Vector3 operator *(float value, Vector3 v) {

            float tx = v.x * value;
            float ty = v.y * value;
            float tz = v.z * value;

            return new Vector3(tx, ty, tz);
        }

        /**
         * Divides a vector with a gives scalar
         */
        public static Vector3 operator /(Vector3 v, float value) {

            if (value <= 0) return new Vector3(0.0f, 0.0f, 0.0f);
            float tx = v.x / value;
            float ty = v.y / value;
            float tz = v.z / value;
            return new Vector3(tx, ty, tz);

        }

        /**
         * Compares two vectors
         * 
         * @return True if the vectors are the same
         */
        public static bool operator == (Vector3 v1, Vector3 v2) {

            if (v1.x == v2.x && v1.y == v2.y && v1.z == v2.z) {
 
                return true;
            }

            return false;

        }

        /**
         * Compares two vectors
         * 
         * @return True if the vectors are not the same
         */
        public static bool operator != (Vector3 v1, Vector3 v2) {

            if (v1.x != v2.x || v1.y != v2.y || v1.z != v2.z) {
            
                return true;
           
            } else {
            
                return false;
            
            }
        }

        /**
         * Returns a copy of the current vector
         */
        public Vector3 Clone () {
        
            return new Vector3(x,y,z);
        
        }

        /**
         * Overridden to get rid of the csc compiler warnings :) 
         */
        public override int GetHashCode() {

            return base.GetHashCode();
        
        }

        /**
         * Compares a given vector to the current vector instance
         * 
         * @return True if the vectors are the same
         */
        public override bool Equals(object obj) {

            if (obj.GetType() != this.GetType()) {         
                return false;
            } else {
                return (this == (Vector3)obj);
            }
        }

        /**
         * Returns a float array, representing the vector
         */
        public float[] Data() {
            return new float[3] { x, y, z };
        }

        /**
         * Returns a float array, representing the vector
         */
        public void Data(float[] v) {

            v[0] = x;
            v[1] = y;
            v[2] = z;
        }
    }
}
